<?php
namespace Rechecker\Tools;

/**
 * Клиент для работы с reChecker API
 *
 * Использование:
 * $client = new ApiClient('ваш_api_ключ');
 * $result = $client->checkMetaTags('https://example.com');
 */
class ApiClient
{
    const API_BASE = 'https://rechecker.ru/api/external';
    const TIMEOUT = 60;

    private string $apiKey;

    public function __construct(string $apiKey = '')
    {
        if (empty($apiKey)) {
            // Если ключ не передан — берём из настроек модуля
            $apiKey = \COption::GetOptionString('rechecker.tools', 'api_key', '');
        }
        $this->apiKey = $apiKey;
    }

    /**
     * Общий метод для POST-запросов к API
     */
    private function post(string $endpoint, array $data): array
    {
        if (empty($this->apiKey)) {
            return ['success' => false, 'error' => 'API ключ не настроен. Укажите ключ в настройках модуля reChecker Tools.'];
        }

        $url = self::API_BASE . '/' . ltrim($endpoint, '/');
        $payload = json_encode($data);

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-API-Key: ' . $this->apiKey,
            ],
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_TIMEOUT => self::TIMEOUT,
            CURLOPT_SSL_VERIFYPEER => true,
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            return ['success' => false, 'error' => 'Ошибка соединения с reChecker API: ' . $curlError];
        }

        $decoded = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return ['success' => false, 'error' => 'Некорректный ответ от API (HTTP ' . $httpCode . ')'];
        }

        return $decoded;
    }

    /**
     * Проверка мета-тегов страницы
     * Возвращает: title, description, h1, h2[], ogTitle, ogDescription, ogImage,
     *             canonical, analysis (h1Count, h2Count, imageCount, linkCount),
     *             recommendations[]
     */
    public function checkMetaTags(string $url): array
    {
        return $this->post('meta-tags', ['url' => $url]);
    }

    /**
     * Проверка цепочки редиректов
     * Возвращает: redirects[], totalRedirects, finalUrl, finalStatus
     */
    public function checkRedirects(string $url): array
    {
        return $this->post('check-redirects', ['url' => $url]);
    }

    /**
     * Проверка SSL-сертификата
     * Возвращает: hostname, grade, ssl (valid, validFrom, validTo, daysRemaining, protocol, ...),
     *             recommendations[]
     */
    public function checkSsl(string $url): array
    {
        return $this->post('ssl', ['url' => $url]);
    }

    /**
     * Анализ HTTP заголовков безопасности
     * Возвращает: grade, score, headers[], allHeaders{}, recommendations[]
     */
    public function checkHeaders(string $url): array
    {
        return $this->post('headers', ['url' => $url]);
    }

    /**
     * Проверка скорости загрузки (Web Vitals)
     * Возвращает: metrics (ttfb, responseTime, pageSize), resources, renderBlocking,
     *             serverInfo, recommendations[]
     */
    public function checkWebVitals(string $url): array
    {
        return $this->post('web-vitals', ['url' => $url]);
    }

    /**
     * Поиск битых ссылок на странице
     * Возвращает: totalChecked, broken, ok, brokenLinks[], links[], recommendations[]
     */
    public function checkBrokenLinks(string $url, int $limit = 50): array
    {
        return $this->post('broken-links', ['url' => $url, 'limit' => $limit]);
    }

    /**
     * Анализ robots.txt
     * Возвращает: exists, url, content, analysis (userAgents, disallowRules, ...),
     *             recommendations[]
     */
    public function checkRobots(string $url): array
    {
        return $this->post('robots', ['url' => $url]);
    }

    /**
     * Проверка sitemap.xml
     * Возвращает: exists, url, urlCount, sitemapCount, isIndex, sampleUrls[],
     *             recommendations[]
     */
    public function checkSitemap(string $url): array
    {
        return $this->post('sitemap', ['url' => $url]);
    }

    /**
     * WHOIS информация о домене
     * Возвращает: domain, whois (creationDate, expirationDate, registrar, nameservers),
     *             ageDays, ageYears, daysUntilExpiry, dns, recommendations[]
     */
    public function checkWhois(string $url): array
    {
        return $this->post('whois', ['url' => $url]);
    }

    /**
     * Анализ изображений на странице
     * Возвращает: images[], analysis (total, withoutAlt, withAlt, external, lazy),
     *             recommendations[]
     */
    public function checkImages(string $url): array
    {
        return $this->post('images', ['url' => $url]);
    }

    /**
     * Проверяет корректность API ключа
     */
    public function validateKey(): bool
    {
        if (empty($this->apiKey)) return false;
        $result = $this->checkMetaTags('https://example.com');
        return isset($result['success']) && $result['success'];
    }
}
